;rapid_flow_map.ncl

;PURPOSE
;This script allows to create several image files (with .png format) from a 
;polyline shapefile with mapped rivers and a corresponding netCDF file with 
;RAPID outputs.  In these flow maps, the thickness of each blue line varies 
;with time as a function of the flow that goes through it.
;REQUIREMENTS:
;    -NCL  (http://www.ncl.ucar.edu/)  
;    -A polyline shapefile with two fields named COMID and RIVID storing the 
;     same unique integers identifying RAPID rivers 
;    -A RAPID output file corresponding to the shapefile
;AUTHOR
;Cedric H. David, 2012


;*******************************************************************************
;Start
;*******************************************************************************
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_code.ncl"
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_csm.ncl"

begin

;-------------------------------------------------------------------------------
;Domain preferences 
;-------------------------------------------------------------------------------
;map_name                = "San_Guad"
;min_lat                 = 28
;max_lat                 = 31
;min_lon                 = -100.2
;max_lon                 = -96.2
;tick_lat                = 2
;tick_lon                = 2
;title                   = "River flow in the San Antonio and Guadalupe Basins, TX"
;shp_file                = "./shp/NHDFlowline_San_Guad_with_dir_anim.shp"
;Qout_file               = "../rapid_unified/output/Qout_San_Guad_1460days_p3_dtR=900s_n1_comid.nc"
;text1                   = "http://www.geo.utexas.edu/scientist/david/rapid.htm"
;text2                   = "David et al. (2011), Journal of Hydrometeorology, DOI: 10.1175/2011JHM1345.1"
;jul_str                 = greg2jul(2004,1,1,0) ;First time step in RAPID output
;kk_str                  = 1    ; 10415;  9200 
;kk_end                  = 11680; 10415; 11100 
;thk_factor              = 100

;map_name                = "France"
;min_lat                 = 40.6
;max_lat                 = 51.6
;min_lon                 = -5.0
;max_lon                 = 11.0
;tick_lat                = 4
;tick_lon                = 4
;title                   = "River flow in the SIM-France domain"
;shp_file                = "./shp/rivsurf_polyline_WGS84.shp"
;Qout_file               = "../rapid_unified/output/Qout_France_201101_c_zvol_ext_3653days_pb_dtR=1800s_nx.nc"
;text1                   = "http://www.geo.utexas.edu/scientist/david/rapid.htm"
;text2                   = "David et al. (2011), Hydrological Processes, DOI: 10.1002/hyp.8070"
;jul_str                 = greg2jul(1995,8,1,0) ;First time step in RAPID output
;kk_str                  = 1    ;22153 
;kk_end                  = 21924;22153
;thk_factor              = 500

;map_name                = "Reg12"
;min_lat                 = 25.8
;max_lat                 = 34.8
;min_lon                 = -104.4
;max_lon                 = -92.4
;tick_lat                = 4
;tick_lon                = 4
;title                   = "River flow in the Texas Gulf Coast Hydrologic Region"
;shp_file                = "./shp/NHDFlowline_Reg12_with_dir_anim.shp"
;Qout_file               = "../rapid_unified/output/Qout_Reg12_2000_2007_Noah_MP_pb0.nc"
;text1                   = "http://www.geo.utexas.edu/scientist/david/rapid.htm"
;text2                   = "David et al. (201x), in preparation"
;jul_str                 = greg2jul(2000,1,1,0) ;First time step in RAPID output
;kk_str                  = 1    ;11689;1    ; 21923; 
;kk_end                  = 23368;14616;23368; 21923;  
;thk_factor              = 150

;map_name                = "Mississippi"
;min_lat                 = 25.5
;max_lat                 = 52.5
;min_lon                 = -113.7
;max_lon                 = -77.7
;tick_lat                = 10
;tick_lon                = 10
;title                   = "River flow in the Mississippi River Basin"
;shp_file                = "./shp/NHDFlowline_Mississippi_THINN1_5_Digitized_anim.shp"
;Qout_file               = "/home/u1/aat669/rapid/output/Qout_mississippi_3652days_nhd_phi2_kfac3.nc"
;text1                   = "http://www.geo.utexas.edu/scientist/david/rapid.htm"
;text2                   = "Tavakoly et al. (201x), in preparation"
;jul_str                 = greg2jul(2000,1,1,0) ;First time step in RAPID output
;kk_str                  = 23857
;kk_end                  = 24592  
;thk_factor              = 2500

map_name                = "Reg07"
min_lat                 = 36.3
max_lat                 = 48.3
min_lon                 = -99.7
max_lon                 = -83.7
tick_lat                = 4
tick_lon                = 4
title                   = "River flow in the Upper Mississippi Basin"
shp_file                = "./shp/NHDFlowline_Reg07_with_dir_anim.shp"
Qout_file               = "../rapid_unified/output/Qout_Reg07_2004_2004_VIC_V4.0.5_pa0_single_n1_preonly_ilu.nc"
text1                   = "http://www.geo.utexas.edu/scientist/david/rapid.htm"
text2                   = "David and Yang (201x), in preparation"
jul_str                 = greg2jul(2004,1,1,0) ;First time step in RAPID output
kk_str                  = 481  ;               ;first time step in animation
kk_end                  = 1456 ;  
thk_factor              = 1125

;-------------------------------------------------------------------------------
;Map settings 
;-------------------------------------------------------------------------------
wks_type="png"
wks_type@wkWidth=640
wks_type@wkHeight=480
;wks_type@wkOrientation="landscape"
wks  = gsn_open_wks(wks_type,map_name)
;wks  = gsn_open_wks("ps",map_name)

res                     = True

res@gsnFrame            = False          ; don't advance frame yet

res@gsnMaximize         = True           ; maximize plot in frame

;res@vpWidthF=3
;res@vpHeightF=2
;res@gsnPanelDebug =True

res@mpDataBaseVersion   = "MediumRes"    ; slightly better resolution base map

res@mpMinLatF           = min_lat
res@mpMaxLatF           = max_lat
res@mpMinLonF           = min_lon
res@mpMaxLonF           = max_lon
;Zoom in on area of interest

res@tiMainString        = title 
res@tiMainFontHeightF   = 0.018   ; Make font slightly smaller.
;Title

txres                   = True
txres@txFontHeightF     = 0.019
;Timestamp

res@gsnMajorLonSpacing  = tick_lon        
res@gsnMajorLatSpacing  = tick_lat
res@tmXBLabelFontHeightF= 0.015
res@tmYLLabelFontHeightF= 0.015
;Tick marks. This one doesn't seem to adjust with plot maximization

;-------------------------------------------------------------------------------
;Read shapefile with rivers
;-------------------------------------------------------------------------------
f = addfile(shp_file, "r")   ; Open shapefile
;Add file

segments = f->segments
geometry = f->geometry
segsDims = dimsizes(segments)  ;Number of lines we'll end up drawing.
;print(segsDims(0))         
geomDims = dimsizes(geometry)
lon      = f->x
lat      = f->y
rivid_shp= f->COMID
;print(rivid(0))         
;Read data off shapefile

geom_segIndex = f@geom_segIndex
geom_numSegs  = f@geom_numSegs
segs_xyzIndex = f@segs_xyzIndex
segs_numPnts  = f@segs_numPnts
numFeatures   = geomDims(0)
;Read global attributes  


;-------------------------------------------------------------------------------
;Read RAPID output file
;-------------------------------------------------------------------------------
q = addfile(Qout_file, "r")   
;variables = getfilevarnames(q)
;print(variables)

rivid_nc = q->COMID
Qout     = q->Qout
;print(rivid_nc(0))         
;print(Qout(11680-1,5175-1))


;-------------------------------------------------------------------------------
;Find index index of nc corresponding to index of shp
;-------------------------------------------------------------------------------
index=rivid_shp
do ii=0, numFeatures-1
     index(ii)=-1
end do
;Not sure how to create a new array, so copied an existing one

;do ii=0, numFeatures-1  
;     print(ii)
;     do jj=0, numFeatures-1  
;          if(rivid_shp(ii) .eq. rivid_nc(jj)) then
;              index(ii)=jj
;          end if
;     end do
;end do
;;The following while loop seems to be at least 2.5 times faster

do ii=0 , numFeatures-1
     ;print(ii)
     jj=0
     ;do while ((rivid_shp(ii) .ne. rivid_nc(jj)) .and. (jj .lt. numFeatures))
     do while ((rivid_shp(ii) .ne. rivid_nc(jj))) ;this one faster but danger
          jj=jj+1
     end do
     index(ii)=jj
end do
;print(rivid_shp(5174))
;print(index(5174))
;print(rivid_nc(index(5174)))

;-------------------------------------------------------------------------------
;Draw rivers 
;-------------------------------------------------------------------------------
lines = new(segsDims(0),graphic)   ; array to hold polylines

plres             = True           ; resources for polylines
plres@gsLineColor = "blue"

segNum = 0       ; Counter for adding polylines

do kk=kk_str-1,kk_end-1

plot = gsn_csm_map(wks,res)   ; Draw map, but don't advance frame.
gsn_text_ndc(wks,text1,0.50,0.07,txres)
gsn_text_ndc(wks,text2,0.50,0.02,txres)

time_array=jul2greg(jul_str+kk/8)
;print(time_array)
time_stamp=sprinti("%04i",time_array(0))+"/"+sprinti("%02i",time_array(1))+"/"+\ 
                   sprinti("%02i",time_array(2));+"-"+tostring(kk)
;print(time_stamp)
gsn_text_ndc(wks,time_stamp,0.80,0.75,txres)

do ii=0, numFeatures-1  

     plres@gsLineThicknessF = 0.5+4.0*(Qout(kk,index(ii))/thk_factor)
     ;plres@gsLineThicknessF = 0.5+4.0*(Qout(kk,ii)/thk_factor)

     startSegment = geometry(ii, geom_segIndex)
     numSegments  = geometry(ii, geom_numSegs)

     do seg=startSegment, startSegment+numSegments-1
          startPT = segments(seg, segs_xyzIndex)
          endPT   = startPT + segments(seg, segs_numPnts) - 1
          gsn_polyline(wks, plot, lon(startPT:endPT),  \
                                  lat(startPT:endPT), plres)

          segNum = segNum + 1
     end do

end do
frame(wks)   ; Advanced frame.

end do

;*******************************************************************************
;End
;*******************************************************************************
end

