#!/bin/bash

#PURPOSE
#This script allows to download one day of land model data from NLDAS2 stored 
#in 24 hourly gridded binary (.grb) files and to convert these to one day of 
#RAPID inputs stored in one netCDF file with 3-hourly inflow to rivers
#REQUIREMENTS:
#    -wget (http://www.gnu.org/software/wget/)
#    -NCL  (http://www.ncl.ucar.edu/)  
#    -NCO  (http://nco.sourceforge.net/)
#AUTHOR
#Cedric H. David, 2012



#*******************************************************************************
#General information
#*******************************************************************************
ftp=ftp://hydro1.sci.gsfc.nasa.gov/data/s4pa/NLDAS
mod=NLDAS_MOS0125_H.002
year=2012
day=001

dir_runoff_grb_1hr=./runoff_grb_1hr
dir_runoff_nc_1hr=./runoff_nc_1hr
dir_m3_riv_nc_1hr=./m3_riv_nc_1hr
dir_m3_riv_nc_3hr=./m3_riv_nc_3hr


#*******************************************************************************
#Create directories if don't exist
#*******************************************************************************
echo "Creating directories if they don't exist"
mkdir -p $dir_runoff_grb_1hr/$mod/$year/$day
mkdir -p $dir_runoff_nc_1hr/$mod/$year/$day
mkdir -p $dir_m3_riv_nc_1hr/$mod/$year/$day
mkdir -p $dir_m3_riv_nc_3hr/$mod/$year/$day


#*******************************************************************************
#Download grb data
#*******************************************************************************
echo "Downloading NLDAS2 hourly runoff data in grb format"
wget -A grb --timeout=20 -t inf -nH --cut-dirs=3                               \
            --directory-prefix=$dir_runoff_grb_1hr                             \
            -r -c -o $dir_runoff_grb_1hr/log                                   \
            $ftp/$mod/$year/$day


#*******************************************************************************
#Convert runoff grb to nc and change variable names in nc files
#*******************************************************************************
echo "Converting NLDAS2 hourly runoff data from grb to nc format,"             \
     "renaming variables"
#-------------------------------------------------------------------------------
#Convert files from grb to nc 
#-------------------------------------------------------------------------------
for file in `ls $dir_runoff_grb_1hr/$mod/$year/$day/*.grb`
do
#echo $file
ncl_convert2nc $file                                                           \
               -v SSRUN_GDS0_SFC_ave2h,BGRUN_GDS0_SFC_ave2h,g0_lon_1,g0_lat_0  \
               -o $dir_runoff_nc_1hr/$mod/$year/$day -B
done 

#-------------------------------------------------------------------------------
#Rename dimensions and variables 
#-------------------------------------------------------------------------------
for file in `ls $dir_runoff_nc_1hr/$mod/$year/$day/*.nc`
do
ncrename -d g0_lon_1,lon                                                       \
         -d g0_lat_0,lat                                                       \
         -v g0_lon_1,lon                                                       \
         -v g0_lat_0,lat                                                       \
         -v SSRUN_GDS0_SFC_ave2h,RUNSF                                         \
         -v BGRUN_GDS0_SFC_ave2h,RUNSB                                         \
         $file 
done

#-------------------------------------------------------------------------------
#End
#-------------------------------------------------------------------------------


#*******************************************************************************
#convert runoff nc file to RAPID nc file
#*******************************************************************************
echo "Converting NLDAS2 hourly runoff data to hourly RAPID inputs"

for runoff_file in `ls $dir_runoff_nc_1hr/$mod/$year/$day/*.nc`
do

#-------------------------------------------------------------------------------
#Create name of m3_file  
#-------------------------------------------------------------------------------
m3_file=$runoff_file
m3_file=${m3_file:(-37)}
m3_file=$dir_m3_riv_nc_1hr/$mod/$year/$day/m3_riv_$m3_file

#echo $runoff_file
#echo $m3_file

#-------------------------------------------------------------------------------
#update runoff file name in fortran coupling program  
#-------------------------------------------------------------------------------
sed -i -e "s|runoff_nc_file     =.*|runoff_nc_file     ='$runoff_file'|"       \
       -e "s|m3_nc_file         =.*|m3_nc_file         ='$m3_file'|"           \
          ./rapid_coupler_1file.f90

#-------------------------------------------------------------------------------
#update runoff file name in fortran coupling program  
#-------------------------------------------------------------------------------
ifort rapid_coupler_1file.f90 -I $TACC_NETCDF_INC -L $TACC_NETCDF_LIB -lnetcdf
./a.out

#-------------------------------------------------------------------------------
#End
#-------------------------------------------------------------------------------
done


#*******************************************************************************
#Compute 3-hourly accumulation and concatenate all netCDF files together
#*******************************************************************************
echo "Computing 3-hourly accumlations and concatenating into one day of RAPID" \
     "inputs"
ncra -O -y ttl $dir_m3_riv_nc_1hr/$mod/$year/$day/m3_*.0[0-2]00.002.nc         \
               $dir_m3_riv_nc_3hr/$mod/$year/$day/accum_0000.nc
ncra -O -y ttl $dir_m3_riv_nc_1hr/$mod/$year/$day/m3_*.0[3-5]00.002.nc         \
               $dir_m3_riv_nc_3hr/$mod/$year/$day/accum_0300.nc
ncra -O -y ttl $dir_m3_riv_nc_1hr/$mod/$year/$day/m3_*.0[6-8]00.002.nc         \
               $dir_m3_riv_nc_3hr/$mod/$year/$day/accum_0600.nc
ncra -O -y ttl $dir_m3_riv_nc_1hr/$mod/$year/$day/m3_*.0900.002.nc             \
               $dir_m3_riv_nc_1hr/$mod/$year/$day/m3_*.1[0-1]00.002.nc         \
               $dir_m3_riv_nc_3hr/$mod/$year/$day/accum_0900.nc
ncra -O -y ttl $dir_m3_riv_nc_1hr/$mod/$year/$day/m3_*.1[2-4]00.002.nc         \
               $dir_m3_riv_nc_3hr/$mod/$year/$day/accum_1200.nc
ncra -O -y ttl $dir_m3_riv_nc_1hr/$mod/$year/$day/m3_*.1[5-7]00.002.nc         \
               $dir_m3_riv_nc_3hr/$mod/$year/$day/accum_1500.nc
ncra -O -y ttl $dir_m3_riv_nc_1hr/$mod/$year/$day/m3_*.1[8-9]00.002.nc         \
               $dir_m3_riv_nc_1hr/$mod/$year/$day/m3_*.2000.002.nc             \
               $dir_m3_riv_nc_3hr/$mod/$year/$day/accum_1800.nc
ncra -O -y ttl $dir_m3_riv_nc_1hr/$mod/$year/$day/m3_*.2[1-3]00.002.nc         \
               $dir_m3_riv_nc_3hr/$mod/$year/$day/accum_2100.nc


ncrcat -O $dir_m3_riv_nc_3hr/$mod/$year/$day/accum*.nc                         \
          $dir_m3_riv_nc_3hr/$mod/$year/$day/concat.nc
rm $dir_m3_riv_nc_3hr/$mod/$year/$day/accum*.nc 

for m3_file in `ls $dir_m3_riv_nc_1hr/$mod/$year/$day/m3_*0000*.nc`
do
m3_file=${m3_file:(-37)}
m3_file=${m3_file/0000/3hrly}
m3_file=$dir_m3_riv_nc_3hr/$mod/$year/$day/m3_riv_$m3_file
#echo $m3_file
done

mv $dir_m3_riv_nc_3hr/$mod/$year/$day/concat.nc $m3_file
