program rapid_coupler_1file
use netcdf
implicit none

!PURPOSE
!This coupler allows to convert runoff information from a land surface model   
!to a volume of water entering RAPID river reaches.  Both files are in netCDF
!format.  A .csv file that connects both and that is produced in a GIS is needed
!in the process.
!INPUTS
!Runoff data are in kg/m2 accumulated over a time step and include:  
!    -Surface runoff in a variable called 'RUNSF'
!    -Subsurface runoff in a variable called 'RUNSB'
!A .csv file is needed as an input that contains for each river reach:
!    -The ID of the each river reach
!    -The area of its contributing catchment in km2
!    -The i index of the grid cell where the contributing catchment centroid   
!     is located within the runoff file
!    -The j index
!OUTPUTS
!Inflow data to RAPID river reaches are in m3 accumulated over a time step:
!    -Inflow in a variable called 'm3_riv'
!NOTES
!Runoff values of 1e20 (NoData) are replaced by 0 in the entire runoff domain
!before calculations of RAPID inputs and their total number is counted.
!This program stops if runoff values over 1000 kg/m2 or under 0 kg/m2 are 
!detected anywhere in the runoff file.  
!This program assumes that each contributing catchment is located on only one
!runoff grid cell which is a valid approximation if catchments are much smaller
!than the grid cell size.
!AUTHOR
!Cedric H. David, 2012 (bug fix by Jong Sung Lee)


!*******************************************************************************
!Declaration of variables
!*******************************************************************************
integer :: IS_lon, JS_lon
integer :: IS_lat, JS_lat

integer :: IS_reachtot, JS_reachtot

integer :: IS_nodata_runsf,IS_nodata_runsb

real, dimension(:), allocatable :: ZV_lon
real, dimension(:), allocatable :: ZV_lat

integer, dimension(:), allocatable :: IV_domain_id
integer, dimension(:), allocatable :: IV_i_index
integer, dimension(:), allocatable :: IV_j_index

real, dimension(:), allocatable :: ZV_areakm

real, dimension(:,:), allocatable :: ZM_runsf
real, dimension(:,:), allocatable :: ZM_runsb

real, dimension(:), allocatable :: ZV_m3_riv  !accumulated over some time

integer :: IS_nc_status
integer :: IS_nc_id_fil_run, IS_nc_id_fil_m3
!netCDF file ids
integer :: IS_nc_id_dim_lon, IS_nc_id_dim_lat, IS_nc_id_dim_time,              &
           IS_nc_id_dim_comid
!netCDF dimension ids
integer, parameter :: IS_nc_ndim_run=2, IS_nc_ndim_rap=2 
!number of dimensions
integer, dimension(IS_nc_ndim_run) ::                   IV_nc_start_run,       &
                                      IV_nc_count_run
!IV_nc_start_run, IV_nc_count_run are in (/lon,lat/) format
integer, dimension(IS_nc_ndim_rap) :: IV_nc_id_dim_rap, IV_nc_start_rap,       &
                                      IV_nc_count_rap
!IV_nc_start_rap, IV_nc_count_rap are in (/COMID,time/) format
integer :: IS_nc_id_var_runsf, IS_nc_id_var_runsb, IS_nc_id_var_m3,            &
           IS_nc_id_var_comid
!netCDF variable ids

character(len=100) :: coupling_file
!Unit 10
character(len=100) :: runoff_nc_file
character(len=100) :: m3_nc_file


!*******************************************************************************
!Current variables
!*******************************************************************************
IS_lon=464
IS_lat=224
runoff_nc_file     ='./runoff_nc_1hr/NLDAS_MOS0125_H.002/2012/001/NLDAS_MOS0125_H.A20120101.2300.002.nc'

IS_reachtot=68143
coupling_file      ='./Reg12_comid_area_i_j_nldas2_whole_domain.csv'
m3_nc_file         ='./m3_riv_nc_1hr/NLDAS_MOS0125_H.002/2012/001/m3_riv_NLDAS_MOS0125_H.A20120101.2300.002.nc'


!*******************************************************************************
!Allocate sizes 
!*******************************************************************************
allocate(ZV_lon(IS_lon))
allocate(ZV_lat(IS_lat))

allocate(IV_domain_id(IS_reachtot))
allocate(IV_i_index(IS_reachtot))
allocate(IV_j_index(IS_reachtot))
allocate(ZV_areakm(IS_reachtot))
allocate(ZV_m3_riv(IS_reachtot))

allocate(ZM_runsf(IS_lon,IS_lat))
allocate(ZM_runsb(IS_lon,IS_lat))


!*******************************************************************************
!Reads coupling_file
!*******************************************************************************
open(10,file=coupling_file,status='old')
do JS_reachtot=1,IS_reachtot
     read(10,*) IV_domain_id(JS_reachtot),ZV_areakm(JS_reachtot),              &
                IV_i_index(JS_reachtot),IV_j_index(JS_reachtot)
end do
close(10)
!print *,IV_domain_id(IS_reachtot),ZV_areakm(IS_reachtot),                      &
!        IV_i_index(IS_reachtot),IV_j_index(IS_reachtot) 
!print *,IV_domain_id(50536),ZV_areakm(50536)            ,                      &
!        IV_i_index(50536),IV_j_index(50536) 


!*******************************************************************************
!Defines netCDF file for RAPID inputs
!*******************************************************************************
IS_nc_status=NF90_CREATE(m3_nc_file,NF90_CLOBBER,IS_nc_id_fil_m3)
call nc_check_status(IS_nc_status)
!Creates netCDF file, NF90_CLOBBER is default: overwrite existing dataset if any

IS_nc_status=NF90_DEF_DIM(IS_nc_id_fil_m3,'COMID',IS_reachtot,                 &
                          IS_nc_id_dim_comid)
call nc_check_status(IS_nc_status)
IS_nc_status=NF90_DEF_DIM(IS_nc_id_fil_m3,'time',NF90_UNLIMITED,               &
                          IS_nc_id_dim_time)
call nc_check_status(IS_nc_status)
!Defines dimensions

IV_nc_id_dim_rap(1)=IS_nc_id_dim_comid
IV_nc_id_dim_rap(2)=IS_nc_id_dim_time
!Assigns dimension IDs in vector of dimension IDs. 
!Unlimited dimension must come last

IS_nc_status=NF90_DEF_VAR(IS_nc_id_fil_m3,'m3_riv',NF90_REAL,                  &
                          IV_nc_id_dim_rap,IS_nc_id_var_m3)
call nc_check_status(IS_nc_status)
IS_nc_status=NF90_DEF_VAR(IS_nc_id_fil_m3,'COMID',NF90_INT,IS_nc_id_dim_comid,IS_nc_id_var_comid)
call nc_check_status(IS_nc_status)
!Defines variable
IS_nc_status=NF90_ENDDEF(IS_nc_id_fil_m3)
call nc_check_status(IS_nc_status)
!End definition 


!*******************************************************************************
!Process runoff netcdf file
!*******************************************************************************
!-------------------------------------------------------------------------------
!Initialize values 
!-------------------------------------------------------------------------------
IV_nc_start_run=(/1,1/)
IV_nc_count_run=(/IS_lon,IS_lat/)

IV_nc_start_rap=(/1,1/)
IV_nc_count_rap = (/IS_reachtot,1/)

do JS_lon=1,IS_lon
do JS_lat=1,IS_lat
     ZM_runsf(JS_lon,JS_lat)=0
     ZM_runsb(JS_lon,JS_lat)=0
end do
end do

IS_nodata_runsf=0
IS_nodata_runsb=0

do JS_reachtot=1,IS_reachtot
     ZV_m3_riv(JS_reachtot) = 0.
end do

!-------------------------------------------------------------------------------
!Read runoff file
!-------------------------------------------------------------------------------
IS_nc_status=NF90_OPEN(runoff_nc_file,NF90_NOWRITE,IS_nc_id_fil_run)
call nc_check_status(IS_nc_status)
          
IS_nc_status=NF90_INQ_VARID(IS_nc_id_fil_run,'RUNSB',IS_nc_id_var_runsf)
call nc_check_status(IS_nc_status)
IS_nc_status=NF90_GET_VAR(IS_nc_id_fil_run,IS_nc_id_var_runsf,ZM_runsf,        &
                          IV_nc_start_run,IV_nc_count_run)
call nc_check_status(IS_nc_status)
IS_nc_status=NF90_INQ_VARID(IS_nc_id_fil_run,'RUNSF',IS_nc_id_var_runsb)
call nc_check_status(IS_nc_status)
IS_nc_status=NF90_GET_VAR(IS_nc_id_fil_run,IS_nc_id_var_runsb,ZM_runsb,        &
                          IV_nc_start_run,IV_nc_count_run)
call nc_check_status(IS_nc_status)

IS_nc_status=NF90_CLOSE(IS_nc_id_fil_run)
call nc_check_status(IS_nc_status)

!-------------------------------------------------------------------------------
!Remove NoData values
!-------------------------------------------------------------------------------
do JS_lon=1,IS_lon
do JS_lat=1,IS_lat
     if (ZM_runsf(JS_lon,JS_lat)==1e20) then
          IS_nodata_runsf=IS_nodata_runsf+1
          ZM_runsf(JS_lon,JS_lat)=0
     end if
     if (ZM_runsb(JS_lon,JS_lat)==1e20) then 
          IS_nodata_runsb=IS_nodata_runsb+1
          ZM_runsb(JS_lon,JS_lat)=0
     end if
end do
end do
!Remove NoData values, here NoData=1e20
!print *, ZM_runsf(228,228),ZM_runsb(228,228)
!print *, ZM_runsf(1,1),ZM_runsb(1,1)

!-------------------------------------------------------------------------------
!Stop if runoff has weird values
!-------------------------------------------------------------------------------
if (maxval(ZM_runsf)>1000) stop 'Surface runoff exceeds 1000'
if (maxval(ZM_runsb)>1000) stop 'Subsurface runoff exceeds 1000'
if (minval(ZM_runsf)<0) stop 'Negative surface runoff'
if (minval(ZM_runsb)<0) stop 'Negative subsurface runoff'

!-------------------------------------------------------------------------------
!Actual coupling procedure
!-------------------------------------------------------------------------------
do JS_reachtot=1,IS_reachtot
     JS_lon=IV_i_index(JS_reachtot)
     JS_lat=IV_j_index(JS_reachtot)
     if (JS_lon == 0 .AND. JS_lat == 0) then
          ZV_m3_riv(JS_reachtot) = 0
     else 
          ZV_m3_riv(JS_reachtot)=( ZM_runsf(JS_lon,JS_lat)                     &
                                  +ZM_runsb(JS_lon,JS_lat))                    &
                                 *ZV_areakm(JS_reachtot)*1000
     end if
     !with runoff in kg/m2=mm and area in km2
end do

!-------------------------------------------------------------------------------
IS_nc_status=NF90_PUT_VAR(IS_nc_id_fil_m3,IS_nc_id_var_m3,ZV_m3_riv,           &
                               IV_nc_start_rap,IV_nc_count_rap)
call nc_check_status(IS_nc_status)

IS_nc_status=NF90_PUT_VAR(IS_nc_id_fil_m3,IS_nc_id_var_comid,IV_domain_id) 
call nc_check_status(IS_nc_status)
!Populate netCDF file


!-------------------------------------------------------------------------------
!end of processing of one file
!-------------------------------------------------------------------------------
IS_nc_status=NF90_CLOSE(IS_nc_id_fil_m3)
call nc_check_status(IS_nc_status)


!*******************************************************************************
!Prints some information
!*******************************************************************************
!print *, 'Runoff file                                 :', runoff_nc_file
!print *, 'Number of NoData points in surface runoff   :', IS_nodata_runsf
!print *, 'Maximum value for surface runoff (kg/m2)    :', maxval(ZM_runsf)
!print *, 'Number of NoData points in subsurface runoff:', IS_nodata_runsb
!print *, 'Maximum value for subsurface runoff (kg/m2) :', maxval(ZM_runsb)
!print *, 'm3_riv file                                 :', m3_nc_file
!print *, 'Maximum value for RAPID inputs (m3)         :', maxval(ZV_m3_riv)


!*******************************************************************************
!end program
!*******************************************************************************
!print *, 'Done. netCDF input file for RAPID created, overwrite if already exist'


!*******************************************************************************
!subroutine
!*******************************************************************************
contains
subroutine nc_check_status(IS_nc_status)

integer, intent ( in) :: IS_nc_status
    
if(IS_nc_status /= nf90_noerr) then 
     print *, trim(nf90_strerror(IS_nc_status))
     stop "Stopped"
end if

end subroutine nc_check_status  
end program rapid_coupler_1file
